<?php

declare(strict_types=1);

namespace Erlage\Photogram\Requests\Hashtag;

use Erlage\Photogram\Constants\ResponseConstants;
use Erlage\Photogram\Exceptions\RequestException;
use Erlage\Photogram\Pattern\ExceptionalRequests;
use Erlage\Photogram\Data\Tables\Sys\RequestTable;
use Erlage\Photogram\Data\Models\Hashtag\HashtagModel;
use Erlage\Photogram\Data\Tables\Hashtag\HashtagTable;
use Erlage\Photogram\Data\Models\Hashtag\HashtagFinder;
use Erlage\Photogram\Data\Tables\Hashtag\HashtagFollowTable;

final class HashtagContent extends ExceptionalRequests
{
    public static function loadSingle(): void
    {
        self::process(function ()
        {
            /*
            |--------------------------------------------------------------------------
            | get data from request
            |--------------------------------------------------------------------------
            */

            $hashtagIdFromReq = self::$request -> findKey(
                HashtagTable::ID,
                RequestTable::PAYLOAD,
                HashtagTable::TABLE_NAME
            );

            if ( ! self::isAvailable($hashtagIdFromReq))
            {
                $hashtagDisplayTextFromReq = self::$request -> findKey(
                    HashtagTable::DISPLAY_TEXT,
                    RequestTable::PAYLOAD,
                    HashtagTable::TABLE_NAME
                );

                self::ensureValue(ResponseConstants::ERROR_BAD_REQUEST_MSG, $hashtagDisplayTextFromReq);

                $useHashtagLiteralDisplayTextValue = true;
            }
            else
            {
                $useHashtagLiteralDisplayTextValue = false;
            }

            /*
            |--------------------------------------------------------------------------
            |  ensure target model exists
            |--------------------------------------------------------------------------
            */

            if ( ! $useHashtagLiteralDisplayTextValue)
            {
                $targetHashtagModel = HashtagModel::findFromId_noException($hashtagIdFromReq);

                self::ensureModel($targetHashtagModel, ResponseConstants::ENTITY_NOT_FOUND_MSG);
            }
            else
            {
                $finder = (new HashtagFinder())
                    -> setDisplayText($hashtagDisplayTextFromReq)
                    -> find();

                if ($finder -> notFound())
                {
                    throw new RequestException(ResponseConstants::ERROR_BAD_REQUEST_MSG);
                }

                $targetHashtagModel = $finder -> popModelFromResults();
            }

            /*
            |--------------------------------------------------------------------------
            |  add model to response
            |--------------------------------------------------------------------------
            */

            self::addToResponse(HashtagTable::getTableName(), $targetHashtagModel -> getDataMap());

            /*
            |--------------------------------------------------------------------------
            |  add additional content iff current user is authenticated
            |--------------------------------------------------------------------------
            */

            if (self::$userSession -> isAuthenticated())
            {
                /*
                |--------------------------------------------------------------------------
                | additional data | hashtag follow map
                | -------------------------------------------------------------------------
                | help build follow button
                |--------------------------------------------------------------------------
                */

                self::fetchModelsAndAddAsAdditional(
                    HashtagFollowTable::getTableName(),
                    array(
                        HashtagFollowTable::FOLLOWED_HASHTAG_ID => $targetHashtagModel -> getId(),
                        HashtagFollowTable::FOLLOWED_BY_USER_ID => self::$authedUserModel -> getId(),
                    )
                );
            }
        });
    }
}
